#region Copyright (C) 2005-2007 Benjamin Schrter <benjamin@irgendwie.net>
//
// This file is part of PhotoTagStudio
//
// PhotoTagStudio is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// PhotoTagStudio is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PhotoTagStudio; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, 5th Floor, Boston, MA 02110-1301 USA.
#endregion

using System;
using System.Collections.Generic;
using System.IO;
using System.Windows.Forms;
using Raccoom.Windows.Forms;
using Schroeter.Photo;
using Schroeter.PhotoTagStudio.About;
using Schroeter.PhotoTagStudio.Properties;
using Schroeter.Windows.Forms;

namespace Schroeter.PhotoTagStudio.Gui
{
    public partial class MainForm : Form, IStatusDisplay
    {
        private PictureDetailControlList displayControls;
        private string navigateTo;
        private string currentDirectory;

        private StatusDisplay mainStatusDisplay;
        private StatusDisplay secondStatusDisplay;
        
        public MainForm(string startDirOrFile) : this()
        {
            navigateTo = startDirOrFile.Trim();
        }

        public MainForm()
        {
            InitializeComponent();

            this.Icon = Resources.PTS;

            this.splitContainer1.Panel1MinSize = 100;
            this.splitContainer1.Panel2MinSize = 400;
            this.splitContainer2.Panel1MinSize = 100;
            this.splitContainer2.Panel2MinSize = 325;
            this.splitContainer3.Panel2MinSize = 100;
            this.splitContainer3.Panel1MinSize = 100;

            displayControls = new PictureDetailControlList();
            displayControls.Add(this.pictureDisplay);
            displayControls.Add(this.iptcEditor1);
            displayControls.Add(this.exifDisplay1);
            displayControls.Add(this.exifgpsEditor1);
            displayControls.Add(this.completeTagList1);
            displayControls.Add(this.renamer1);
            displayControls.Add(this.imageDisplay1);        

            displayControls.RegisterEvents(this.RefreshDetailViews, 
                                           this.RefreshCurrentDirectoryView, 
                                           this.RefreshDirectoryTree, 
                                           this.GoToNextFile,
                                           this.ListAllCheckedFiles, 
                                           this.ListSelectedDirectory);

            mainStatusDisplay = new StatusDisplay(this.toolStripProgressBar1, this.toolStripStatusLabel1);

            secondStatusDisplay = new StatusDisplay(this.toolStripProgressBar2);
            this.pictureDisplay.SetStatusDisplay(secondStatusDisplay);
        }

        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            Settings.Default.LastWorkingDirectory = currentDirectory;
            Settings.Default.Save();
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            tableLayoutPanel1_Resize(sender, e);

            //TODO TreeViewFolderBrowserDataProvider ds = new Raccoom.Windows.Forms.TreeViewFolderBrowserDataProvider();
            TreeViewFolderBrowserDataProviderShell32 dss32 = new TreeViewFolderBrowserDataProviderShell32();
            this.directoryTree.DataSource = dss32;

            this.directoryTree.Populate();
            this.directoryTree.Nodes[0].Expand();

            this.fileList.BindSelectAllCheckBox(this.chkSelectAllFiles);

            this.displayControls.RefreshSettings();

            if (this.navigateTo != "")
            {
                FileInfo fi = new FileInfo(this.navigateTo);
                if (fi.Exists)
                {
                    this.directoryTree.ShowFolder(fi.Directory.FullName);
                    this.directoryTree.SelectedDirectories.Add(fi.Directory.FullName);
                    this.directoryTree.SelectedNode.Expand();

                    foreach (TreeNode n in this.fileList.Nodes)
                        if (n.Tag is FileInfo)
                            if (((FileInfo)n.Tag).Name.ToUpper() == fi.Name.ToUpper())
                            {
                                this.fileList.SelectedNode = n;
                                break;
                            }
                }
                else
                {
                    DirectoryInfo di = new DirectoryInfo(this.navigateTo);
                    if (di.Exists)
                    {
                        this.directoryTree.ShowFolder(di.FullName);
                        this.directoryTree.SelectedDirectories.Add(di.FullName);
                        this.directoryTree.SelectedNode.Expand();
                    }
                }
            }

            this.buttonViewPreview.Checked = Settings.Default.GuiShowPreview;
            this.buttonViewFolderTree.Checked = Settings.Default.GuiShowDirectoryTree;

            MainForm_Resize(sender, e);
        }

        private void RefreshFileList(string path)
        {
            MainForm_FormClosed(null, null);
            
            //this.fileList.BeginUpdate();
            this.fileList.Nodes.Clear();
            RefreshFileInfo(null);

            if (!path.StartsWith("::") && path != "")
            {
                DirectoryInfo di = new DirectoryInfo(path);
                if (di.Exists)
                {
                    currentDirectory = path;
                    
                    FileInfo[] files = di.GetFiles("*.jpg");
                    Array.Sort(files, CompareFileInfo);

                    foreach (FileInfo fi in files)
                    {
                        ThreeStateTreeNode n = new ThreeStateTreeNode(fi.Name);
                        n.CheckState = CheckState.Checked;
                        n.Tag = fi;
                        this.fileList.Nodes.Add(n);
                    }

                    
                }
            }

            this.fileList.UpdateSelectAllCheckBox();

            //ShowFileCount
            this.labFileCount.Text = String.Format("{0} files", this.fileList.Nodes.Count);


            //this.fileList.EndUpdate();
        }

        private static int CompareFileInfo(FileInfo a, FileInfo b)
        {
            return a.Name.CompareTo(b.Name);
        }

        private void RefreshFileInfo(FileInfo fi)
        {
            PictureMetaData pmd;

            if (fi == null)
                pmd = null;
            else
                pmd = new PictureMetaData(fi.FullName);

            this.displayControls.UpdatePicture(pmd);
        }

        private void RefreshDetailViews()
        {
            this.displayControls.RefreshSettingsAndData();
        }
        
        private void GoToNextFile()
        {
            TreeNode n = this.fileList.SelectedNode;
            int i = this.fileList.Nodes.IndexOf(n);
            if ( i+1 != this.fileList.Nodes.Count )
            {
                n = this.fileList.Nodes[i+1];
                this.fileList.SelectedNode = n;
            }
            
        }

        private void RefreshCurrentDirectoryView()
        {
            TreeNodePath node = this.directoryTree.SelectedNode as TreeNodePath;
            if (node != null) this.RefreshFileList(node.Path);		
        }

        private void RefreshDirectoryTree(string navigateTo)
        {
            if (navigateTo != "")
            {
                this.directoryTree.Populate();
                this.directoryTree.ShowFolder(navigateTo);
                this.directoryTree.SelectedDirectories.Add(navigateTo);
            }

            this.RefreshFileList(navigateTo);
        }

        private void directoryTree_AfterSelect(object sender, TreeViewEventArgs e)
        {
            TreeNodePath node = e.Node as TreeNodePath;
            DirectoryInfo di = null;
            if (node != null)
                di = new DirectoryInfo(node.Path);
                
            if ( di != null && di.Exists )
            {
                this.RefreshFileList(di.FullName);
                this.displayControls.UpdateDirectory(di.FullName);
            }
            else
                MessageBox.Show("The directory does no longer exist!", this.Text, MessageBoxButtons.OK, MessageBoxIcon.Error);

        }

        private void directoryTree_BeforeSelect(object sender, TreeViewCancelEventArgs e)
        {
            TreeNodePath node = e.Node as TreeNodePath;
            if (node != null)
                if (node.Path.StartsWith("::"))
                    e.Cancel = true;
        }


        private void fileList_AfterSelect(object sender, TreeViewEventArgs e)
        {
            FileInfo fi = null;

            if (e.Node != null)
                fi = e.Node.Tag as FileInfo;

            if (fi != null && fi.Exists)
                RefreshFileInfo(fi);
            else
                MessageBox.Show("The file does not longer exist!", this.Text, MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private List<string> ListAllCheckedFiles(bool subdirectories)
        {
            List<string> l = new List<string>();
            foreach (ThreeStateTreeNode n in this.fileList.Nodes)
                if (n.CheckState == CheckState.Checked && n.Tag is FileInfo)
                    l.Add(((FileInfo)n.Tag).FullName);

            if ( subdirectories)
            {
                TreeNodePath node = this.directoryTree.SelectedNode as TreeNodePath;
                if (node != null)
                {
                    Queue<DirectoryInfo> dirs = new Queue<DirectoryInfo>();
                    foreach(DirectoryInfo d in (new DirectoryInfo(node.Path)).GetDirectories())
                        dirs.Enqueue(d);

                    while (dirs.Count > 0)
                    {
                        DirectoryInfo dir = dirs.Dequeue();
                        if (dir.Exists)
                        {
                            FileInfo[] files = dir.GetFiles("*.jpg");
                            foreach (FileInfo file in files)
                                l.Add(file.FullName);
                            foreach (DirectoryInfo subdir in dir.GetDirectories())
                                dirs.Enqueue(subdir);
                        }
                    }
                }
            }

            return l;
        }

        private List<string> ListSelectedDirectory(bool subdirectories)
        {
            TreeNodePath node = this.directoryTree.SelectedNode as TreeNodePath;
            if (node != null)
            {
                string path = node.Path;
                List<string> outputList = new List<string>();
                Queue<DirectoryInfo> queue = new Queue<DirectoryInfo>();
                queue.Enqueue(new DirectoryInfo(path));

                while (queue.Count > 0)
                {
                    DirectoryInfo dir = queue.Dequeue();
                    if (dir.Exists)
                    {
                        outputList.Insert(0,dir.FullName);
                        if ( subdirectories )
                            foreach (DirectoryInfo d in dir.GetDirectories())
                                queue.Enqueue(d);
                    }
                }

                return outputList;
            }
            else
                return new List<string>();
        }
        
        #region Menu - set parts visible
        private void viewPreview_Click(object sender, EventArgs e)
        {
            this.buttonViewPreview.Checked = !this.buttonViewPreview.Checked;
        }
        private void viewFolder_Click(object sender, EventArgs e)
        {
            this.buttonViewFolderTree.Checked = !this.buttonViewFolderTree.Checked;
        }
        private void viewPreview_CheckedChanged(object sender, EventArgs e)
        {
            Settings.Default.GuiShowPreview = this.buttonViewPreview.Checked;
            this.splitContainer3.Panel2Collapsed = !this.buttonViewPreview.Checked;
            this.buttonViewPreview.Checked = this.buttonViewPreview.Checked;
        }
        private void viewFolder_CheckedChanged(object sender, EventArgs e)
        {
            Settings.Default.GuiShowDirectoryTree = this.buttonViewFolderTree.Checked;
            this.splitContainer1.Panel1Collapsed = !this.buttonViewFolderTree.Checked;
            this.buttonViewFolderTree.Checked = this.buttonViewFolderTree.Checked;
        }
        #endregion

        private void menuFileExit_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void menuViewSettings_Click(object sender, EventArgs e)
        {
            SettingsForm f = new SettingsForm();
            f.ShowDialog(this);


            // todo: nur wenn sich was verndert hat
            this.displayControls.RefreshSettingsAndData();
        }

        private void menuHelpAbout_Click(object sender, EventArgs e)
        {
            AboutBox a = new AboutBox();
            a.ShowDialog();
        }

        public void WaitCursor(bool on)
        {
            this.Cursor = on ? Cursors.WaitCursor : Cursors.Default;
        }

        #region IStatusDisplay
        public void WorkStart(int partCounts)
        {
            WaitCursor(true);
            
            mainStatusDisplay.WorkStart(partCounts);
        }

        public void WorkNextPart()
        {
            mainStatusDisplay.WorkNextPart();
        }
        
        public void WorkFinished()
        {
            mainStatusDisplay.WorkFinished();

            WaitCursor(false);
        }
        #endregion

        #region F5 = refresh
        private void fileList_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.F5)
            {
                if (CanChangeFile())
                {
                    TreeNodePath node = this.directoryTree.SelectedNode as TreeNodePath;
                    if (node != null)
                    {
                        this.WaitCursor(true);

                        string file = "";
                        if (this.fileList.SelectedNode != null
                             && this.fileList.SelectedNode.Tag is FileInfo)
                            file = ((FileInfo)this.fileList.SelectedNode.Tag).Name;

                        this.RefreshFileList(node.Path);

                        foreach (TreeNode n in this.fileList.Nodes)
                            if (n.Tag is FileInfo)
                                if (((FileInfo)n.Tag).Name == file)
                                {
                                    this.fileList.SelectedNode = n;
                                    break;
                                }

                        this.WaitCursor(false);
                    }
                }
            }
        }

        private void directoryTree_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.F5)
            {
                this.WaitCursor(true);

                DirectoryInfo d = null;
                TreeNodePath node = this.directoryTree.SelectedNode as TreeNodePath;
                if (node != null)
                    if (!node.Path.StartsWith("::") && node.Path != "")
                        d = new DirectoryInfo(node.Path);
                
                this.directoryTree.Populate();

                if (d != null && d.Exists)
                {
                    this.directoryTree.ShowFolder(d.FullName);
                    this.directoryTree.SelectedDirectories.Add(d.FullName);
                    this.directoryTree.SelectedNode.Expand();
                    this.RefreshFileList(d.FullName);
                }
                else
                {
                    this.directoryTree.Nodes[0].Expand();
                    this.RefreshFileList("");
                }

                this.WaitCursor(false);
            }
        }
        #endregion

        private void menuExtrasMassTagging_Click(object sender, EventArgs e)
        {
            string path = "";
            TreeNodePath node = this.directoryTree.SelectedNode as TreeNodePath;
            if ( node != null && !node.Path.StartsWith("::"))
                path = node.Path;

            Features.MassTagging.MassTaggingForm f = new Features.MassTagging.MassTaggingForm(path);
            f.ShowDialog(this);

            if (f.SettingsChanged)
                this.displayControls.RefreshSettingsAndData();
        }

        private void fileList_BeforeSelect(object sender, TreeViewCancelEventArgs e)
        {
            if (!CanChangeFile())
                e.Cancel = true;
        }

        private bool CanChangeFile()
        {
            RequestFileChangeResult r = this.displayControls.CanUpdatePicture();

            if (r == RequestFileChangeResult.DoNotClose)
                return false;

            if (r == RequestFileChangeResult.CloseAndSave)
                this.displayControls.CurrentPicture.SaveChanges();

            return true;
        }

        private void menuFileOpenDirectory_Click(object sender, EventArgs e)
        {
            if (!CanChangeFile())
                return;

            FolderBrowserDialog f = new FolderBrowserDialog();
            f.SelectedPath = currentDirectory;
            if (f.ShowDialog(this) == DialogResult.OK)
            {
                this.directoryTree.ShowFolder(f.SelectedPath );
                this.directoryTree.SelectedDirectories.Clear();
                this.directoryTree.SelectedDirectories.Add(f.SelectedPath);
                this.directoryTree.SelectedNode.Expand();
            }
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            this.pictureDisplay.StopAllWork();
            
            if (!CanChangeFile())
                e.Cancel = true;
        }

        private void tableLayoutPanel1_Resize(object sender, EventArgs e)
        {
            int h = this.tableLayoutPanel1.GetRowHeights()[0]-10;
            this.fileList.Height = h - (h % this.fileList.ItemHeight)+3;
        }

        private void menuHelpLicense_Click(object sender, EventArgs e)
        {
            AboutAndInfo.ShowLicence();
        }

        private void MainForm_Resize(object sender, EventArgs e)
        {
            Padding p = this.toolStripProgressBar2.Margin;
            p.Left = this.Width - 130;
            this.toolStripProgressBar2.Margin = p;
        }

    }
}