#region Copyright (C) 2005-2007 Benjamin Schrter <benjamin@irgendwie.net>
//
// This file is part of PhotoTagStudio
//
// PhotoTagStudio is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// PhotoTagStudio is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PhotoTagStudio; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, 5th Floor, Boston, MA 02110-1301 USA.
#endregion

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using Schroeter.Photo;
using Schroeter.PhotoTagStudio.Properties;

namespace Schroeter.PhotoTagStudio.Gui
{
    public partial class Renamer : PictureDetailControlBase
    {
        public Renamer()
        {
            InitializeComponent();

            InitHelpString();
        }

        protected override void RefreshMyData()
        {
            UpdatePreviews();
        } 
        protected override void ClearMyData()
        {
            UpdatePreviews();
        }
        protected override void RefreshMySettings()
        {
            this.txtFilename.DataSource = null;
            this.txtFilename.DataSource = Settings.Default.FilenameFormats.Data;
            this.txtFilename.SelectedItem = Settings.Default.FilenameFormats.DefaultValue;

            this.txtDirectoryname.DataSource = null;
            this.txtDirectoryname.DataSource = Settings.Default.DirectorynameFormats.Data;
            this.txtDirectoryname.SelectedItem = Settings.Default.DirectorynameFormats.DefaultValue;

            this.chkFilename.Checked = true;
            this.chkDirectory.Checked = false;
        }
        private void UpdatePreviews()
        {
            if (currentPicture != null)
            {
                string s;

                FormatFilename(currentPicture, false, out s);
                this.labFilenameExample.Text = s + ".jpg";

                FormatFilename(currentPicture, true, out s);
                this.labDirectoryExample.Text = s;
            }
        }

        #region Formating
        private string FormatDate(DateTime d, string format)
        {
            try
            {
                return d.ToString(format);
            }
            catch (Exception)
            {
                return "";
            }
        }
        
        private bool FormatFilename(PictureMetaData currentPicture, bool formatDirectory, out string newFileName)
        {
            bool renamed = false;
            string s = formatDirectory ? this.txtDirectoryname.Text : this.txtFilename.Text;

            #region text fields
            if (s.Contains("%cap") && currentPicture.IptcCaption != "")
                renamed = true;
            s = s.Replace("%cap", currentPicture.IptcCaption);

            if (currentPicture.ListContact().Count > 0)
            {
                if (s.Contains("%con") && currentPicture.ListContact()[0] != "")
                    renamed = true;
                s = s.Replace("%con", currentPicture.ListContact()[0]);
            }
            else
                s = s.Replace("%con", "");

            if (s.Contains("%on") && currentPicture.IptcObjectName != "")
                renamed = true;
            s = s.Replace("%on", currentPicture.IptcObjectName);

            if (s.Contains("%cr") && currentPicture.IptcCopyright != "")
                renamed = true;
            s = s.Replace("%cr", currentPicture.IptcCopyright);

            if (s.Contains("%sl") && currentPicture.IptcSubLocation != "")
                renamed = true;
            s = s.Replace("%sl", currentPicture.IptcSubLocation);

            if (s.Contains("%cn") && currentPicture.IptcCountryName != "")
                renamed = true;
            s = s.Replace("%cn", currentPicture.IptcCountryName);

            if (s.Contains("%cc") && currentPicture.IptcCountryCode != "")
                renamed = true;
            s = s.Replace("%cc", currentPicture.IptcCountryCode);

            if (s.Contains("%h") && currentPicture.IptcHeadline != "")
                renamed = true;
            s = s.Replace("%h", currentPicture.IptcHeadline);

            if (s.Contains("%c") && currentPicture.IptcCity != "")
                renamed = true;
            s = s.Replace("%c", currentPicture.IptcCity);

            if (s.Contains("%s") && currentPicture.IptcProvinceState != "")
                renamed = true;
            s = s.Replace("%s", currentPicture.IptcProvinceState);

            if (currentPicture.ListByline().Count > 0)
            {
                if (s.Contains("%a") && currentPicture.ListByline()[0] != "")
                    renamed = true;
                s = s.Replace("%a", currentPicture.ListByline()[0]);
            }
            else
                s = s.Replace("%a", "");

            if (currentPicture.ListWriter().Count > 0)
            {
                if (s.Contains("%w") && currentPicture.ListWriter()[0] != "")
                    renamed = true;
                s = s.Replace("%w", currentPicture.ListWriter()[0]);
            }
            else
                s = s.Replace("%w", "");

            string keywords = "";
            foreach (string keyword in currentPicture.ListKeyword())
                keywords += keyword + ", ";
            if (keywords.Length > 0)
                keywords = keywords.Substring(0, keywords.Length - 2);
            if (s.Contains("%k") && keywords != "")
                renamed = true;            
            s = s.Replace("%k", keywords);
            #endregion

            #region date time fields
            renamed |= ReplaceDateFields("%dc", currentPicture.ExifOriginalDateTime, ref s);
            renamed |= ReplaceDateFields("%dd", currentPicture.ExifDigitizedDateTime, ref s);
            renamed |= ReplaceDateFields("%d", currentPicture.IptcDateTimeCreated, ref s);
            #endregion

            s = s.Replace('\\', ' ');
            s = s.Replace(':', ' ');
            s = s.Replace('/', ' ');

            newFileName = s.Trim();
            if ( newFileName.Length == 0 )
                return false;

            return renamed;
        }

        private bool ReplaceDateFields(string field, DateTime? value, ref string s)
        {
            bool renamed = false;

            int fieldLen = field.Length;
            
            while (s.Contains(field))
            {
                renamed = true;

                string format = "";
                int pos = s.IndexOf(field);
                int posStart = s.IndexOf("[", pos);
                int posEnd = -1;
                if (posStart != -1 )
                    posEnd = s.IndexOf("]", posStart);
                if (pos != -1 && posStart != -1 && posEnd != -1 && pos + fieldLen == posStart)
                {
                    format = s.Substring(posStart+1, posEnd - posStart - 1);
                    s = s.Remove(posStart , posEnd - posStart+1 );
                }

                s = s.Remove(pos, fieldLen);
                if (value.HasValue)
                    s = s.Insert(pos, FormatDate(value.Value, format));
            }
            return renamed;
        }
        #endregion

        #region update help
        private void InitHelpString()
        {
            StringBuilder b = new StringBuilder();

            b.Append("%h\tHeadline\r\n");
            b.Append("%cap\tCaption\r\n");
            b.Append("%on\tObject name\r\n");
            b.Append("%w\tWriter\r\n");
            b.Append("%a\tAuthor (byline)\r\n");
            b.Append("%cr\tCopyright\r\n");
            b.Append("%con\tContact\r\n");
            b.Append("%c\tCity\r\n");
            b.Append("%sl\tSublocation\r\n");
            b.Append("%s\tProvince / state\r\n");
            b.Append("%cn\tCountry name\r\n");
            b.Append("%cc\tCountry code\r\n");
            b.Append("%k\tKeywords (comma seperated)\r\n");
            b.Append("\r\n");
            b.Append("%d\tIPTC created date time\r\n");
            b.Append("%dc\tEXIF created date time\r\n");
            b.Append("%dd\tEXIF digitized date time\r\n");
            b.Append("\r\n");
            b.Append("For the date time fields you can specify optional a format.\r\n");
            b.Append("Use [ and ] with the following patterns after the field name (e.g. %dc[yyyy-MM-dd])\r\n");
            b.Append("\r\n");
            b.Append("dd\tThe day of the month. Single-digit days will have a leading zero.\r\n");
            b.Append("ddd \tThe abbreviated name of the day of the week.\r\n");
            b.Append("dddd \tThe full name of the day of the week.\r\n");
            b.Append("MM \tThe numeric month. Single-digit months will have a leading zero.\r\n");
            b.Append("MMM \tThe abbreviated name of the month.\r\n");
            b.Append("MMMM \tThe full name of the month.\r\n");
            b.Append("yy \tThe year without the century. \r\n");
            b.Append("yyyy \tThe year in four digits.\r\n");
            b.Append("hh\tThe hour in a 12-hour clock. Single-digit hours will have a leading zero.\r\n");
            b.Append("HH \tThe hour in a 24-hour clock. Single-digit hours will have a leading zero.\r\n");
            b.Append("mm \tThe minute. Single-digit minutes will have a leading zero.\r\n");
            b.Append("ss \tThe second. Single-digit seconds will have a leading zero.\r\n");
            b.Append("\r\n");
            b.Append("d \tShort date pattern.\r\n");
            b.Append("D \tLong date pattern.\r\n");
            b.Append("f \tFull date and time pattern (long date and short time).\r\n");
            b.Append("F \tFull date and time pattern (long date and long time).\r\n");
            b.Append("g \tGeneral pattern (short date and short time).\r\n");
            b.Append("G \tGeneral pattern (short date and long time).\r\n");
            b.Append("m, M \tMonth day pattern.\r\n");
            b.Append("r, R \tRFC1123 pattern.\r\n");
            b.Append("s \tSortable date time pattern (based on ISO 8601).\r\n");
            b.Append("t \tShort time pattern.\r\n");
            b.Append("T \tLong time pattern.\r\n");
            b.Append("u \tUniversal sortable date time pattern using the format for universal time display.\r\n");
            b.Append("U \tFull date and time pattern (long date and long time) using universal time.\r\n");
            b.Append("y, Y \tYear month pattern.");

            this.richTextBox1.Text = b.ToString();
        }
        #endregion

        #region TextChanged-Events
        private void txtFilename_TextChanged(object sender, EventArgs e)
        {
            this.UpdatePreviews();
            this.chkFilename.Checked = true;
        }
        private void txtDirectoryname_TextChanged(object sender, EventArgs e)
        {
            this.UpdatePreviews();
            this.chkDirectory.Checked = true;
        }
        private void txtDateFormat_TextChanged(object sender, EventArgs e)
        {
            this.UpdatePreviews();
        }
        #endregion

        private void btnRename_Click(object sender, EventArgs e)
        {
            IStatusDisplay statusDisplay = (IStatusDisplay)this.FindForm();

            StopOtherWorker();
            
            List<string> filenames;
            if (this.chkFilename.Checked)
                filenames = this.GetAllFileList(this.chkSubdirs.Checked);
            else
                filenames = new List<string>();
            List<string> directories;
            if (this.chkDirectory.Checked)
                directories = this.GetAllDirectoryList(this.chkSubdirs.Checked);
            else
                directories = new List<string>();

            statusDisplay.WorkStart(filenames.Count + directories.Count);

            #region renaming files
            filenames.Sort();
            foreach (string filename in filenames)
            {
                try
                {
                    FileInfo fi = new FileInfo(filename);
                    if (fi.Exists)
                    {
                        PictureMetaData pmd = new PictureMetaData(filename);
                        string newname;
                        if (FormatFilename(pmd, false, out newname))
                        {
                            newname = fi.DirectoryName + "\\" + newname;

                            string completenewname = newname + fi.Extension;
                            if (completenewname.ToLower() != filename.ToLower())
                            {
                                int i = 1;
                                while (File.Exists(completenewname))
                                    completenewname = newname + " (" + (i++) + ")" + fi.Extension;

                                fi.MoveTo(completenewname);
                            }
                        }

                        pmd.Close();
                    }
                    statusDisplay.WorkNextPart();
                }
                catch (Exception ex)
                {
                    ErrorHandler.LogException(ex, filename);
                }
            }

            Settings.Default.FilenameFormats.AddIfGrowing(this.txtFilename.Text);
            #endregion

            string goToDir = "";
            #region renaming directories
            directories.Sort();
            foreach (string directory in directories)
            {
                try
                {
                    DirectoryInfo di = new DirectoryInfo(directory);
                    if (di.Exists)
                    {
                        PictureMetaData pmd;
                        bool dontClosePmd = false;
                        if (this.currentDirectory == di.FullName && this.currentPicture != null)
                        {
                            pmd = this.currentPicture;
                            dontClosePmd = true;
                        }
                        else
                            pmd = GetPictureMetaDataFromDirectory(di);

                        if (pmd != null)
                        {
                            string newname;
                            if (FormatFilename(pmd, true, out newname))
                            {
                                newname = di.Parent.FullName + "\\" + newname;
                                string completenewname = newname;

                                if (completenewname.ToLower() != directory.ToLower())
                                {
                                    int i = 1;
                                    while (Directory.Exists(completenewname))
                                        completenewname = newname + " (" + (i++) + ")";

                                    di.MoveTo(completenewname);
                                }
                                goToDir = completenewname;
                            }
                            else
                                goToDir = directory;

                            if ( !dontClosePmd )
                                pmd.Close();
                        }
                    }

                    statusDisplay.WorkNextPart();
                }
                catch (Exception ex)
                {
                    ErrorHandler.LogException(ex, directory);
                }
            }

            Settings.Default.DirectorynameFormats.AddIfGrowing(this.txtDirectoryname.Text);
            #endregion

            if (this.chkDirectory.Checked)
                this.FireDirectoryNameChanged(goToDir);
            else
                this.FireDirectoryChanged();

            statusDisplay.WorkFinished();
        }

        private PictureMetaData GetPictureMetaDataFromDirectory(DirectoryInfo di)
        {
            FileInfo[] fis = di.GetFiles("*.jpg");
            if (fis.Length > 0)
            {
                return new PictureMetaData(fis[0].FullName);
            }
            else
                return null;
        }

    }
}