#region Copyright (C) 2005-2007 Benjamin Schrter <benjamin@irgendwie.net>
//
// This file is part of PhotoTagStudio
//
// PhotoTagStudio is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// PhotoTagStudio is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PhotoTagStudio; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, 5th Floor, Boston, MA 02110-1301 USA.
#endregion

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Windows.Forms;
using Schroeter.Photo;
using Schroeter.PhotoTagStudio.Data;
using Schroeter.PhotoTagStudio.Features.KmzMaker;
using Schroeter.PhotoTagStudio.Gui;
using Schroeter.PhotoTagStudio.Properties;
using Schroeter.PhotoTagStudio.Workers;

namespace Schroeter.PhotoTagStudio
{
    public class IptcGpsController : PictureDetailControllerBase
    {
        private IptcView iptcView;
        private ExifGpsView exifGpsView;

        private bool goToNextFileAfterSave;
        private bool processFilesInSubdirectories;
        
        private bool askForFileSave;
        private string lastGpsFile;

        public IptcGpsController(MainForm form, IptcView iptcView, ExifGpsView exifGpsView) : base(form)
        {
            this.iptcView = iptcView;
            this.iptcView.EnterPressed += new EventHandler(Save_Click);

            this.exifGpsView = exifGpsView;
            this.exifGpsView.EnterPressed += new EventHandler(Save_Click);
            this.exifGpsView.CreateKmzClick += new EventHandler(exifGpsView_CreateKmzClick);
            this.exifGpsView.TagFromGpsLogClick += new EventHandler(exifGpsView_TagFromGpsLogClick);

            this.GoToNextFileAfterSave = Settings.Default.GoToNextPicture;
        }

        public override RequestFileChangeResult RequestFileChange()
        {
            if (!askForFileSave)
                return RequestFileChangeResult.Close;
            if (this.currentPicture == null)
                return RequestFileChangeResult.Close;
            if (!this.currentPicture.ExistFile)
                return RequestFileChangeResult.Close;
            IptcModel iptcModel = iptcView.GetModel();
            ExifGpsModel exifGpsModel = exifGpsView.GetModel();
            if (!iptcModel.PendingUpdates && !exifGpsModel.PendingUpdates)
                return RequestFileChangeResult.Close;
            if (!isDataLoaded)
                return RequestFileChangeResult.Close;

            DialogResult r = MessageBox.Show("Save changes to this picture?", "Iptc and Gps Editor", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);

            if (r == DialogResult.Cancel)
                return RequestFileChangeResult.DoNotClose;

            if (r == DialogResult.Yes)
            {
                mainForm.WaitCursor(true);
                SaveToPicture(currentPicture, false);
                mainForm.WaitCursor(false);
                return RequestFileChangeResult.CloseAndSave;
            }

            askForFileSave = false;
            return RequestFileChangeResult.Close;
        }

        #region save
        public void Save_Click(object sender, EventArgs e)
        {
            if ( currentPicture == null )
                return;

            mainForm.WaitCursor(true);

            SaveToPicture(currentPicture, true);

            askForFileSave = false;
            FireDataChanged();

            if (this.goToNextFileAfterSave)
                FireRequestetGoToNext();

            mainForm.WaitCursor(false);            
        }

        private bool SaveToPicture(PictureMetaData pic, bool commit)
        {
            // todo: die worker und modelle knnen mehrfach verwendet werden! 
            IptcWorker iptcWorker = new IptcWorker();
            ExifGpsWorker exifGpsWorker = new ExifGpsWorker();

            IptcModel iptcModel = iptcView.GetModel();
            ExifGpsModel exifGpsModel = exifGpsView.GetModel();

            bool somethingChanged = false;
            if ( iptcWorker.ProcessFile(pic,iptcModel) )
                somethingChanged = true;
            if (exifGpsWorker.ProcessFile(pic,exifGpsModel) )
                somethingChanged = true;
            
            if ( iptcModel.PendingUpdates )
                iptcModel.UpdateSettings();
            
            if ( somethingChanged && commit )
                if (!pic.SaveChanges())
                    return this.ShowFileVanishedMsg(pic.Filename);

            return true;
        }

        public void SaveToAll_Click(object sender, EventArgs e)
        {
            IStatusDisplay statusDisplay = mainForm;
            List<string> filenames = this.GetAllFileList(this.processFilesInSubdirectories);

            PauseOtherWorker();

            statusDisplay.WorkStart(filenames.Count);

            foreach (string filename in filenames)
            {
                PictureMetaData pmd;
                if (this.currentPicture != null
                    && this.currentPicture.Filename == filename)
                    pmd = currentPicture;
                else
                {
                    if (File.Exists(filename))
                        pmd = new PictureMetaData(filename);
                    else
                    {
                        statusDisplay.WorkNextPart();
                        continue;
                    }
                }

                bool breakForeach = SaveToPicture(pmd, true) == false;

                if (pmd != currentPicture)
                    pmd.Close();

                if (breakForeach)
                    break;

                statusDisplay.WorkNextPart();
            }

            FireDataChanged();

            statusDisplay.WorkFinished();

            RestartOtherWorker();
        }
        #endregion

        #region Refresh/Clear Data and Settings
        public override void ClearData()
        {
            base.ClearData();

            iptcView.SetModel(new IptcModel());
            exifGpsView.SetModel(new ExifGpsModel());
        }

        public override void RefreshData()
        {
            base.RefreshData();

            iptcView.SetModel(new IptcModel(currentPicture));
            exifGpsView.SetModel(new ExifGpsModel(currentPicture));

            askForFileSave = true;
        }

        public override void RefreshSettings()
        {
            base.RefreshSettings();

            iptcView.RefreshSettings();
        }
        #endregion

        #region the gps buttons
        void exifGpsView_TagFromGpsLogClick(object sender, EventArgs e)
        {
            // a picture is need for the offset dialog
            if (this.currentPicture == null)
                return;

            // the filename of the gps log
            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Title = "Open gps nmea file";
            ofd.Filter = "*.txt|*.txt";
            if (ofd.ShowDialog(mainForm) != DialogResult.OK)
                return;

            this.lastGpsFile = ofd.FileName;

            // read the gps log
            GpsLog log = GpsLogFactory.FromFile(ofd.FileName);

            // empty log -> exit
            if (log.Count == 0)
                return;

            // ask for gps time offset
            PictureGpsOffsetDialog d = new PictureGpsOffsetDialog(currentPicture, log.FirstTime.Value, log.LastTime.Value);
            d.Offset = Settings.Default.GPSTimeOffset;
            if (d.ShowDialog(mainForm) != DialogResult.OK)
                return;

            // set offset to gps log
            Settings.Default.GPSTimeOffset = d.Offset;
            log.Offset = d.Offset;

            IStatusDisplay statusDisplay = mainForm;

            List<string> filenames = this.GetAllFileList(false);
            statusDisplay.WorkStart(filenames.Count);

            // compute the timespan of all pictures
            DateTime firstPicture = new DateTime(4000, 1, 1);
            DateTime lastPicture = new DateTime(1, 1, 1);
            List<PictureMetaData> pictures = new List<PictureMetaData>();
            foreach (string filename in filenames)
            {
                PictureMetaData pmd;
                if (this.currentPicture != null
                    && this.currentPicture.Filename == filename)
                    pmd = currentPicture;
                else
                    pmd = new PictureMetaData(filename);

                if (pmd.ExifOriginalDateTime.HasValue)
                {
                    DateTime time = pmd.ExifOriginalDateTime.Value;

                    if (time > lastPicture)
                        lastPicture = time;
                    if (time < firstPicture)
                        firstPicture = time;

                    pictures.Add(pmd);
                }
                statusDisplay.WorkNextPart();
            }
            statusDisplay.WorkFinished();

            // ask the user: do it now?
            string text =
                String.Format(
                    "The GPS time is between {0} and {1}.\nThe picture time is between {2} and {3} ({4} and {5}).\n\nContinue?",
                    log.FirstTime,
                    log.LastTime,
                    firstPicture.Add(log.Offset),
                    lastPicture.Add(log.Offset),
                    firstPicture,
                    lastPicture);
            if (MessageBox.Show(text, "Continue?", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
            {
                statusDisplay.WorkStart(pictures.Count);
                foreach (PictureMetaData pmd in pictures)
                {
                    UpdateGpsData(pmd, log);

                    if (pmd != currentPicture)
                        pmd.Close();
                    statusDisplay.WorkNextPart();
                }

                FireDataChanged();
                statusDisplay.WorkFinished();
            }
            else
            {
                foreach (PictureMetaData pmd in pictures)
                    if (pmd != currentPicture)
                        pmd.Close();
            }
        }

        private void UpdateGpsData(PictureMetaData picture, GpsLog log)
        {
            if (picture.ExifOriginalDateTime.HasValue)
            {
                GpsLogEntry entry = log.GetNearestEntry(picture.ExifOriginalDateTime.Value);
                if (entry != null)
                {
                    picture.GpsLatitude = new GpsCoordinate(entry.Latitude);
                    picture.GpsLongitude = new GpsCoordinate(entry.Longitude);
                    picture.GpsLatitudeRef = entry.Latitude >= 0 ? "N" : "S";
                    picture.GpsLongitudeRef = entry.Longitude >= 0 ? "E" : "W";
                    picture.GpsDateTimeStamp = entry.Time;

                    picture.SaveChanges();
                }
            }
        }

        void exifGpsView_CreateKmzClick(object sender, EventArgs e)
        {
            KmzMakerForm f = new KmzMakerForm(currentPicture, currentDirectory, new GetAllFilesDelegate(this.GetAllFileList));
            f.GpsRouteFile = lastGpsFile;
            if (f.ShowDialog(mainForm) == DialogResult.OK)
                if (f.GpsRouteFile != "")
                    lastGpsFile = f.GpsRouteFile;
        }
        #endregion

        #region properties
        public bool GoToNextFileAfterSave
        {
            get { return goToNextFileAfterSave; }
            set
            {
                goToNextFileAfterSave = value;
                Settings.Default.GoToNextPicture = goToNextFileAfterSave;
            }
        }
        public bool ProcessFilesInSubdirectories
        {
            get { return processFilesInSubdirectories; }
            set { processFilesInSubdirectories = value; }
        }
        #endregion
    }
}
