#region Copyright (C) 2005-2007 Benjamin Schrter <benjamin@irgendwie.net>
//
// This file is part of PhotoTagStudio
//
// PhotoTagStudio is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// PhotoTagStudio is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PhotoTagStudio; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, 5th Floor, Boston, MA 02110-1301 USA.
#endregion

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Threading;
using Schroeter.Photo;

namespace Schroeter.PhotoTagStudio
{
    public class ThumbnailCache
    {
        private Dictionary<string, Image> images;
        private IStatusDisplay statusDisplay;

        private string currentPath;
        private Thread batchWorker;
        private Thread singleWorker;
        private Semaphore sema;

        private Size thumbnailSize;

        public ThumbnailCache()
        {
            this.sema = new Semaphore(1, 1);
            images = new Dictionary<string, Image>();}

        #region properties
        public Size ThumbnailSize
        {
            get { return thumbnailSize; }
            set { thumbnailSize = value; }
        }

        public string CurrentPath
        {
            get { return currentPath; }
            set { currentPath = value; }
        }

        #endregion

        public void SetStatusDisplay(IStatusDisplay statusDisplay)
        {
            this.statusDisplay = statusDisplay;
        }

        private bool ExistThumbnail(string name)
        {
            return images.ContainsKey(name.ToLower());
        }
    
        public void CreateAllThumbnails()
        {
            CreateAllThumbnails(CurrentPath);
        }
        public void CreateAllThumbnails(string path)
        {
            Cancel();

            images = new Dictionary<string, Image>();
            CurrentPath = path;

            ReCreateAllThumbnails();
        }

        public void ReCreateAllThumbnails()
        {
            this.sema = new Semaphore(1, 1);
            statusDisplay.WorkFinished();

            batchWorker = new Thread(this.BatchWorkerDoWork);
            batchWorker.Name = "Thumbnail Batch Worker";
            batchWorker.Priority = ThreadPriority.Lowest;
            batchWorker.Start();
        }

        public bool Cancel()
        {
            bool b = false;

            string s = "killing: ";
            
            if (batchWorker != null && batchWorker.IsAlive)
            {
                s += "batch ";
                batchWorker.Abort();
                batchWorker.Join();
                b = true;
            }
            if (singleWorker != null && singleWorker.IsAlive)
            {
                s += "single ";
                singleWorker.Abort();
                singleWorker.Join();
                b = true;
            }

            Console.WriteLine(s);
            
            statusDisplay.WorkFinished();
            
            return b;
        }

        private void BatchWorkerDoWork()
        {
            if (CurrentPath.StartsWith("::"))
                return;
            
            DirectoryInfo di = new DirectoryInfo(CurrentPath);
            FileInfo[] files = di.GetFiles("*.jpg");
            statusDisplay.WorkStart(files.Length);
            foreach (FileInfo fi in files)
            {
                if (!ExistThumbnail(fi.FullName) )
                {
                    CreateThumbnail(fi.FullName);
                }
                else
                {
                    Image i = images[fi.FullName.ToLower()];
                    if (i.Width < thumbnailSize.Width && i.Height < thumbnailSize.Height)
                    {
                        CreateThumbnail(fi.FullName);    
                    }                    
                }

                statusDisplay.WorkNextPart();
            }

            statusDisplay.WorkFinished();
        }

        public delegate void CreateThumbnailCallback(string name, Image thumbnail);
        public bool GetThumbnail(string name, CreateThumbnailCallback callback)
        {
            if ( ExistThumbnail(name) )
            {
                Image i = images[name.ToLower()];
                if (i.Width < thumbnailSize.Width && i.Height < thumbnailSize.Height)
                {
                    // 1st callback with wrong (lower) size
                    callback(name, i);
                    
                    // then generate the new (better) thumbnail
                    if ( singleWorker != null && singleWorker.IsAlive )
                        singleWorker.Abort();
                    singleWorkerName = name;
                    singleWorkerCallback = callback;  // and do a 2nd callback with the better thumbnail with a higher size
                    singleWorker = new Thread(this.SingleWorkerDoWork);
                    singleWorker.Name = "Thumbnail single worker";
                    singleWorker.Priority = ThreadPriority.BelowNormal;
                    singleWorker.Start();

                    return true;
                }
                else
                {
                    callback(name, i);
                    return false;
                }
            }
            else
            {
                callback(name, CreateThumbnail(name));
                return false;
            }
        }

        private string singleWorkerName;
        private CreateThumbnailCallback singleWorkerCallback;        
        private void SingleWorkerDoWork()
        {
            Image i = CreateThumbnail(singleWorkerName);
            singleWorkerCallback(singleWorkerName, i);
        }
        
        private Image CreateThumbnail(string name)
        {           
            name = name.ToLower();
            Image i = null;
            Image iRes = null;
            try
            {
                FileStream fs = new FileStream(name, FileMode.Open, FileAccess.Read);
                i = Image.FromStream(fs);

                double pHeight = (double)i.Height / thumbnailSize.Height;
                double pWidth = (double)i.Width / thumbnailSize.Width;
                if (pHeight > pWidth)
                    iRes = ImageResize.ConstrainProportions(i, thumbnailSize.Height, ImageResize.Dimensions.Height);
                else
                    iRes = ImageResize.ConstrainProportions(i, thumbnailSize.Width, ImageResize.Dimensions.Width);

                fs.Close();
                fs.Dispose();

                sema.WaitOne();
                if (images.ContainsKey(name))
                    images[name] = iRes;
                else
                    images.Add(name, iRes);
                sema.Release();
            }
            catch
            {}
            
            return iRes;
        }
        
        public void Clear()
        {
            images.Clear();
        }
    }
}