#region Copyright (C) 2005-2007 Benjamin Schrter <benjamin@irgendwie.net>
//
// This file is part of PhotoTagStudio
//
// PhotoTagStudio is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// PhotoTagStudio is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PhotoTagStudio; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, 5th Floor, Boston, MA 02110-1301 USA.
#endregion

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using Schroeter.Photo;
using Schroeter.PhotoTagStudio.Data;
using Schroeter.PhotoTagStudio.Features.Renamer;
using Schroeter.PhotoTagStudio.Gui;
using Schroeter.PhotoTagStudio.Properties;
using Schroeter.PhotoTagStudio.Workers;

namespace Schroeter.PhotoTagStudio
{
    class RenameController : PictureDetailControllerBase
    {
        private RenameView renameView;
        private bool processFilesInSubdirectories;

        public RenameController(MainForm form, RenameView renameView) : base(form)
        {
            this.renameView = renameView;
            this.renameView.TextBoxChanged += new EventHandler(renameView_TextBoxChanged);
        }

        public override void ClearData()
        {
            base.ClearData();

            UpdatePreviews();
        }
        public override void RefreshData()
        {
            base.RefreshData();

            UpdatePreviews();
        }
        public override void RefreshSettings()
        {
            base.RefreshSettings();

            this.renameView.RefreshSettings();
        }
        private void UpdatePreviews()
        {
            if (currentPicture != null)
            {
                renameView.labFilenameExample.Text = FileNameFormater.FormatFilename(currentPicture, renameView.txtFilename.Text) + ".jpg";
                renameView.labDirectoryExample.Text = FileNameFormater.FormatFilename(currentPicture, renameView.txtDirectoryname.Text);
            }
        }
        private void renameView_TextBoxChanged(object sender, EventArgs e)
        {
            this.UpdatePreviews();
        }

        #region the actual renaming
        public void Rename_Click(object sender, EventArgs e)
        {            
            // get the model
            RenameModel model = renameView.GetModel();

            // do gui / settings work
            renameView.SaveLastModel();
            Settings.Default.FilenameFormats.AddIfGrowing(model.FilenamePattern);
            Settings.Default.DirectorynameFormats.AddIfGrowing(model.DirectoryPattern);
           
            StopOtherWorker();

            // find files and dirs to process
            List<string> filenames;
            filenames = this.GetAllFileList(this.processFilesInSubdirectories);

            IStatusDisplay statusDisplay = mainForm;
            statusDisplay.WorkStart(filenames.Count);

            // create worker
            RenameWorker worker = new RenameWorker();
            worker.currentPicture = this.currentPicture;
            worker.rootDirectory = this.currentDirectory;
            worker.SetFileList(filenames);
            worker.OneFileProcessed += delegate { statusDisplay.WorkNextPart(); };

            // start working
            worker.ProcessFiles(model);

            // go to the renamd directory or fire events
            if (model.ChangeDirectorynames)
            {
                if (worker.rootDirectory != "" && Directory.Exists((worker.rootDirectory)))
                    this.FireDirectoryNameChanged(worker.rootDirectory);
                else 
                    this.FireDirectoryChanged(true);
            }
            else
                this.FireDirectoryChanged(false);

            FireDataChanged();

            statusDisplay.WorkFinished();
        }
        #endregion

        public bool ProcessFilesInSubdirectories
        {
            get { return processFilesInSubdirectories; }
            set { processFilesInSubdirectories = value; }
        }
    }
}
